<?php
/**
 * Step controller
 * 
 * @package WPFunnels\Rest\Controllers
 */
namespace WPFunnels\Rest\Controllers;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WPFunnels\Wpfnl;
use WPFunnels\Wpfnl_functions;
use WPFunnels\TemplateLibrary\Manager;

class StepController extends Wpfnl_REST_Controller
{

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wpfunnels/v1';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'steps';

	/**
	 * Check if user has valid permission
	 *
	 * @param $request
	 * 
	 * @return bool|WP_Error
	 * @since  1.0.0
	 */
	public function update_items_permissions_check($request)
	{
		if (!Wpfnl_functions::wpfnl_rest_check_manager_permissions( 'steps', 'edit' )) {
			return new WP_Error('wpfunnels_rest_cannot_edit', __('Sorry, you cannot edit this resource.', 'wpfnl'), ['status' => rest_authorization_required_code()]);
		}
		return true;
	}

	/**
	 * Makes sure the current user has access to READ the settings APIs.
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * 
	 * @return WP_Error|boolean
	 * @since  3.0.0
	 */
	public function get_items_permissions_check($request)
	{
		if (!Wpfnl_functions::wpfnl_rest_check_manager_permissions('settings')) {
			return new WP_Error('wpfunnels_rest_cannot_view', __('Sorry, you cannot list resources.', 'wpfnl'), ['status' => rest_authorization_required_code()]);
		}
		return true;
	}


	/**
	 * Register rest routes
	 *
	 * @since 1.0.0
	 */
	public function register_routes()
	{
		register_rest_route($this->namespace, '/' . $this->rest_base, array(
			'args' => array(
				'funnel_id' => array(
					'description' => __('Funnel ID.', 'wpfnl'),
					'type' => 'string',
				),
				'step_id' => array(
					'description' => __('Step ID.', 'wpfnl'),
					'type' => 'string',
				)
			),
			array(
				'methods' => \WP_REST_Server::EDITABLE,
				'callback' => [
					$this,
					'update_step_meta'
				],
				'permission_callback' => [
					$this,
					'update_items_permissions_check'
				] ,
			)
		));
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<step_id>[\d\.]+)/delete-step',
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array(
						$this,
						'delete_step',
					),
					'permission_callback' => array(
						$this,
						'update_items_permissions_check',
				),
			),
		);
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/create-step',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array(
						$this,
						'create_step',
					),
					'permission_callback' => array(
						$this,
						'update_items_permissions_check',
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/wpfunnel-import-step',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array(
						$this,
						'import_step',
					),
					'permission_callback' => array(
						$this,
						'update_items_permissions_check',
					),
				),
			)
		);

		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/update-conditional-status',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array(
						$this,
						'update_conditional_status',
					),
					'permission_callback' => array(
						$this,
						'update_items_permissions_check',
					),
					'args'                => array(
						'stepId'     => array(
							'description' => __( 'Funnel step ID.','wpfnl' ),
							'type'        => 'integer',
							'required'      => true
						),
						'status'     => array(
							'description' => __( 'Condition status.','wpfnl' ),
							'type'        => 'string',
							'required'      => true
						),
					),
				),
			)
		);
		
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/get-conditions',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array(
						$this,
						'get_conditions',
					),
					'permission_callback' => array(
						$this,
						'update_items_permissions_check',
					),
					'args'                => array(
						'stepId'     => array(
							'description' => __( 'Funnel step ID.','wpfnl' ),
							'type'        => 'integer',
							'required'      => true
						)
					),
				),
			)
		);
		
		register_rest_route(
			$this->namespace, '/' . $this->rest_base . '/save-condition',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array(
						$this,
						'save_condition',
					),
					'permission_callback' => array(
						$this,
						'update_items_permissions_check',
					),
					'args'                => array(
						'stepId'     => array(
							'description' => __( 'Funnel step ID.','wpfnl' ),
							'type'        => 'integer',
							'required'      => true
						),
						'conditions'     => array(
							'description' => __( 'Funnel step ID.','wpfnl' ),
							'type'        => 'array',
							'required'      => true
						),
					),
				),
			)
		);
	}

	/**
	 * Responsible to create a single step of funnels
	 *
	 * @param WP_REST_Request $payload Data from request.
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public function create_step( $payload ) {
		if ( ! isset( $payload['funnel_id'] , $payload['step_type'] ) ){
			return $this->prepare_wp_error_response(
				'rest_invalid_request',
				__( 'Invalid rest request.', 'wpfnl' ),
				array( 'status' => 400 )
			);
		}

		$funnel_id = $payload['funnel_id'];
		$step_type = $payload['step_type'];
		$step_name = isset( $payload['step_name'] ) ? $payload['step_name'] : $step_type;
		$funnel    = Wpfnl::get_instance()->funnel_store;
		$step      = Wpfnl::get_instance()->step_store;
		$step_id   = $step->create_step( $funnel_id, $step_name, $step_type );

		$step->set_id( $step_id );

		if ( ! $step_id || is_wp_error( $step_id ) ) {
			return $this->prepare_wp_error_response(
				'rest_step_create_failed',
				__( 'Failed to create a step', 'wpfnl' ),
				array( 'status' => 400 )
			);
		}
		
		$funnel->set_id( $funnel_id );
		$step_edit_link = get_edit_post_link( $step_id );
		$step_view_link = get_post_permalink( $step_id );
		$response = array(
			'success'               => true,
			'step_id'               => $step_id,
			'step_edit_link'        => $step_edit_link,
			'step_view_link'        => rtrim( $step_view_link, '/' ),
			'step_title'            => $step->get_title(),
			'conversion'            => 0,
			'visit'                 => 0,
			'shouldShowAnalytics'   => false,
			'abTestingSettingsData' => [],
		);

		if( isset( $payload['lastClickedAddStep'] ) ){
            $reconfigureSettings = get_post_meta( $funnel->get_id(),'_wpfnl_reconfigurable_condition_data', true);
            if( is_array($reconfigureSettings) && !empty($reconfigureSettings) ){
                $key = array_search($payload['lastClickedAddStep'], array_column($reconfigureSettings, 'nodeId'));
                if( false !== $key ){
                    $reconfigureSettings[$key]['step_id'] = $step_id;
                    update_post_meta( $funnel->get_id(),'_wpfnl_reconfigurable_condition_data', $reconfigureSettings );
                    $response['reconfigureSettings'] = $reconfigureSettings;
                    update_post_meta( $step_id, '_wpfnl_maybe_enable_condition', $reconfigureSettings[$key]['is_enable'] );
                }
            }
        }
		
		return rest_ensure_response( $response );
	}


	/**
	 * Update step meta
	 * 
	 * @param WP_REST_Request $request
	 * 
	 * @return \WP_REST_Response
	 *
	 * @since 2.0.5
	 */
	public function update_step_meta( WP_REST_Request $request ) {

		$step_id = $request['step_id'] ? $request['step_id'] : '';
		$funnel_id = $request['funnel_id'] ? $request['funnel_id'] : '';
		$slug = $request['slug'] ? $request['slug'] : '';
		$settings = $request->get_params();
		$parent_id = get_post_meta( $step_id, '_parent_step_id', true );
		wp_update_post([
			"ID" 			=> $step_id,
			"post_title" 	=> wp_strip_all_tags( $settings['title'] ),
			"post_name" 	=> sanitize_title($settings['slug']),
		]);

		if ( defined('WPFNL_PRO_VERSION') ) {
			do_action( 'wpfunnels/before_update_step_meta', $step_id, $funnel_id, $settings );
		}else{
			if( isset($settings['funnel_id'], $settings['step_id'], $settings['title'], $settings['slug']) ){
				$steps = Wpfnl_functions::get_steps($settings['funnel_id']);
				foreach($steps as $key=>$step){
					if($step['id'] == $settings['step_id']){
						$steps[$key]['name'] = $settings['title'];
					}
				}
				update_post_meta( $settings['funnel_id'], '_steps_order',$steps);
			}
		}
		
		if( $step_id == $parent_id || !$parent_id ){
			$funnel_id= $settings['funnel_id'];
			$funnel_data = get_post_meta($funnel_id,'_funnel_data', true);
			if( isset( $funnel_data['drawflow']['Home']['data']) and is_array($funnel_data['drawflow']['Home']['data']) ){
				
				foreach( $funnel_data['drawflow']['Home']['data'] as $key=> $data ){
					
					if(isset($data['data']['step_id']) && (int)$step_id === (int)$data['data']['step_id'] ){
						
						$edit_post_link = base64_encode(get_edit_post_link( $step_id ));
						$view_link		= base64_encode(get_the_permalink( $step_id ));
						$funnel_data['drawflow']['Home']['data'][$key]['data']['step_view_link'] = $view_link;
						$funnel_data['drawflow']['Home']['data'][$key]['data']['step_edit_link'] = $edit_post_link;
					}
				}
				update_post_meta($funnel_id,'_funnel_data', $funnel_data);
			}
			
			

			$funnel_data = get_post_meta($funnel_id,'funnel_data', true);
			if( isset( $funnel_data['drawflow']['Home']['data']) and is_array($funnel_data['drawflow']['Home']['data']) ){
				foreach( $funnel_data['drawflow']['Home']['data'] as $key=> $data ){
					if(isset($data['data']['step_id']) && (int)$step_id === (int)$data['data']['step_id'] ){
						$edit_post_link = base64_encode(get_edit_post_link( $step_id ));
						$view_link		= base64_encode(get_the_permalink( $step_id ));
						$funnel_data['drawflow']['Home']['data'][$key]['data']['step_view_link'] = $view_link;
						$funnel_data['drawflow']['Home']['data'][$key]['data']['step_edit_link'] = $edit_post_link;
					}
				}
				update_post_meta($funnel_id,'funnel_data', $funnel_data);
			}
			
		}
			
		$first_step_id = Wpfnl_functions::get_first_step( $funnel_id );
		$utm_settings = Wpfnl_functions::get_funnel_utm_settings( $funnel_id );
	    $view_link    = get_post_permalink( $first_step_id );
       
	    if( !empty( $utm_settings['utm_enable'] ) && 'on' === $utm_settings['utm_enable'] ) {
		    unset($utm_settings['utm_enable']);
		    $view_link = add_query_arg($utm_settings,$view_link);
		    $view_link   = strtolower($view_link);
	    }

		$response = array(
			'success'		=> true,
			'post_title'	=> htmlspecialchars_decode(get_the_title($step_id)),
			'permalink'		=> rtrim( get_the_permalink($step_id), '/' ),
			'funnel_main_link'		=> esc_url( $view_link ),
			'slug'			=> sanitize_title($slug),
		);
		return $this->prepare_item_for_response( $response, $request );
		
	}

	/**
	 * Common function to update step meta
	 * 
	 * @param WP_REST_Request $request
	 * 
	 * @return \WP_REST_Response
	 *
	 * @since 2.7.10
	 */
	public static function update_step_meta_on_funnel_name_change( $funnel_id, $step_id, $settings ){
		if( is_plugin_active( 'wpfunnels-pro/wpfnl-pro.php' ) && Wpfnl_functions::is_pro_license_activated() ){
			/**
			 * Fires to update step slug on ab testing variation on funnel name change
			 *
			 * @since 2.7.0
			 *
			 * @param int    $step_id  The ID of the step.
			 * @param int    $funnel_id  The ID of the funnel to which the step belongs.
     		 * @param array  $settings   An array of settings containing the title and slug for the step.
			 */
			do_action( 'wpfunnels/before_update_step_meta_on_funnel_name_change', $step_id, $funnel_id, $settings );
		}else{
			if( isset($settings['funnel_id'], $settings['step_id'], $settings['title'], $settings['slug']) ){
				$steps = Wpfnl_functions::get_steps($settings['funnel_id']);
				if ( !empty( $steps ) ){
					foreach($steps as $key=>$step){
						if($step['id'] == $settings['step_id']){
							$steps[$key]['name'] = $settings['title'];
						}
					}
				}
				update_post_meta( $settings['funnel_id'], '_steps_order', $steps);
				wp_update_post([
					"ID" 			=> $step_id,
					"post_title" 	=> wp_strip_all_tags( $settings['title'] ),
					"post_name" 	=> sanitize_title($settings['slug']),
				]);
			}
		}
	}


	/**
	 * Delete step and all its data
	 *
	 * @param WP_REST_Request $payload Data from request.
	 *
	 * @return array
	 *
	 * @since 2.7.9
	 */
	public function delete_step( $payload ) {
		if ( ! isset ( $payload['step_id'] )){
			return $this->prepare_wp_error_response(
				'rest_invalid_stepID',
				__( 'Invalid Step ID.', 'wpfnl' ),
				array( 'status' => 404 )
			);
		}

		
		$step_id   = sanitize_text_field( $payload['step_id'] );
		$step_type = get_post_meta( $step_id, '_step_type', true );

		$result =  [
			'success' 	=> true,
		];

		if( !empty($payload['nodeId']) ){
			$is_enable 	= Wpfnl_functions::maybe_enable_condition( $step_id );
			$condiitons = Wpfnl_functions::get_conditions( $step_id );
			$next_step  = Wpfnl_functions::get_conditional_next_step( $step_id );
			$funnel_id  = Wpfnl_functions::get_funnel_id_from_step( $step_id );
			$data = get_post_meta( $funnel_id, '_wpfnl_reconfigurable_condition_data', true );
			
			if( !is_array($data) || empty($data) ){
				$data = [];
			}else{
				$key = array_search($payload['nodeId'], array_column($data, 'nodeId'));
				if( false !== $key ){
					unset($data[$key]);
				}
			}

			$is_enable_condition = get_post_meta( $step_id, '_wpfnl_maybe_enable_condition', true );

			if( !empty($payload['stepId']) && !empty($payload['nodeType']) && 'addstep' !== $payload['nodeType']  ){
				update_post_meta( $payload['stepId'], '_wpfnl_maybe_enable_condition', $is_enable_condition );
			}
			
			$updatedData = [
				'step_id' 		=> !empty($payload['nodeType']) && 'addstep' !== $payload['nodeType'] && !empty($payload['stepId']) ? $payload['stepId'] : '',
				'nodeId' 		=> $payload['nodeId'],
				'is_enable'  	=> $is_enable_condition,
				'conditions' 	=> get_post_meta( $step_id, '_wpfnl_step_conditions', true ),
				'next_step' 	=> get_post_meta( $step_id, '_wpfnl_next_step_after_condition', true ),
			];

			array_push( $data, $updatedData );
	
			update_post_meta( $funnel_id, '_wpfnl_reconfigurable_condition_data', $data );
			$result['reconfigurable_condition_data'] = $data;
		}



		/**
		 * Delete the automation data linked with the step
		 *
		 * @since 2.7.0
		 *
		 * @param int   $step_id  Step ID.
		 */
		do_action( 'wpfunnels/before_delete_step', $step_id ); //phpcs:ignore

		$response = $this->prepare_wp_error_response(
			'rest_invalid_stepID',
			__( 'Invalid Step ID.', 'wpfnl' ),
			array( 'status' => 404 )
		);

		if ( $step_type ) {
			$response = $this->delete_regular_step($step_id);
		} else if ( $step_id ){ // For conditional step.
			$response = $this->conditional_step_response($step_id);
		}
		
		$result['response'] = $response;
		return $result;
	}

	/**
	 * Delete a regular step and all its data.
	 *
	 * @param int $step_id The ID of the regular step to be deleted.
	 *
	 * @return array The response containing the success status and message.
	 *
	 * @since 2.7.9
	 */
	private function delete_regular_step( $step_id ) {
		if ( !$step_id ){
			return $this->prepare_wp_error_response(
				'rest_invalid_stepID',
				__( 'Invalid Step ID.', 'wpfnl' ),
				array( 'status' => 404 )
			);
		}
		
		$step = new Wpfnl::$instance->step_store();
		$step->read( $step_id );

		$funnel_id     = $step->get_funnel_id();
		$delete_result = $step->delete( $step_id );
		$funnel        = Wpfnl::$instance->funnel_store;
		$funnel->read($funnel_id);

		if ( $delete_result ) {
			return $this->prepare_wp_success_response( 'Step deleted successfully', 200 );
		} 

		$response = $this->prepare_wp_error_response(
			'rest_delete_failure',
			__( 'Failed to delete step.', 'wpfnl' ),
			array( 'status' => 500 )
		);
		
		return $response ;
	}

	/**
	 * Response for deleting a conditional step.
	 *
	 * @param int $step_id The ID of the conditional step to be deleted.
	 *
	 * @return array The response indicating the success of the deletion.
	 *
	 * @since 2.7.9
	 */
	private function conditional_step_response( $step_id ) {
		if ( !$step_id ){
			return $this->prepare_wp_error_response(
				'rest_invalid_stepID',
				__( 'Invalid Step ID.', 'wpfnl' ),
				array( 'status' => 404 )
			);
		}

	    return $this->prepare_wp_success_response( 'Conditional step deleted successfully',200 );
	}


	/**
	 * Prepare a single setting object for response.
	 *
	 * @param object $item Setting object.
	 * @param WP_REST_Request $request Request object.
	 * 
	 * @return \WP_REST_Response $response Response data.
	 * @since  1.0.0
	 */
	public function prepare_item_for_response($item, $request)
	{
		$data = $this->add_additional_fields_to_object($item, $request);
		return rest_ensure_response( $data );
	}

	/**
     * Import wp funnel steps from remote servers.
     *
     * @param array $payload
     * 
     * @return array
     * @since  1.0.0
     */
    public function import_step( $payload ) {
		$manager_class = new Manager();
        $source		   = $manager_class->get_source(!empty( $payload['source'] ) ? $payload['source'] : 'remote' );
        return $source->import_step($payload);
    }


	/**
	 * Update conditional status for a step.
	 *
	 * This function updates the conditional status for a specific step based on the provided
	 * step ID and status in the request. It checks for required parameters, sanitizes input,
	 * and updates the corresponding post meta with the new status.
	 *
	 * @since 2.9.0
	 *
	 * @param WP_REST_Request $request The REST request object containing parameters.
	 *
	 * @return \WP_REST_Response The REST response indicating the success of the operation.
	 */
	public function update_conditional_status( $request ) {
		$required_params = array( 'stepId', 'status' );

		// Check if all required parameters are present in the request.
		foreach ( $required_params as $param ) {
			if ( ! $request->has_param( $param ) ) {
				return rest_ensure_response( $this->prepare_wp_error_response(
					__( "Required parameter '$param' is missing.", 'wpfnl' ), 400
				) );
			}
		}

		// Get the step ID and status from the request.
		$step_id = sanitize_text_field( $request['stepId'] );
		$status  = sanitize_text_field( $request['status'] );

		if( !$step_id ){
			return rest_ensure_response( $this->prepare_wp_error_response(
				__( "Required parameter step id is empty.", 'wpfnl' ), 400
			) );
		}

		// Update the conditional status post meta.
		update_post_meta( $step_id, '_wpfnl_maybe_enable_condition', $status );

		$funnel_id = get_post_meta( $step_id, '_funnel_id', true );
		$reconfigureSettings = get_post_meta( $funnel_id,'_wpfnl_reconfigurable_condition_data', true);
		if( is_array($reconfigureSettings) && !empty($reconfigureSettings) ){
			$key = array_search($step_id, array_column($reconfigureSettings, 'step_id'));
			if( false !== $key ){
				unset($reconfigureSettings[$key]);
				update_post_meta( $funnel_id,'_wpfnl_reconfigurable_condition_data', $reconfigureSettings );
			}
		}

		// Return a REST response indicating success.
		return rest_ensure_response( [ 'success' => true ] );
	}


	/**
	 * Get conditional status and conditions for a step.
	 *
	 * This function retrieves the conditional status and associated conditions for a specific step
	 * based on the provided step ID in the request. It checks for required parameters, sanitizes input,
	 * and returns the conditional status, along with the array of conditions.
	 *
	 * @since 2.9.0
	 *
	 * @param WP_REST_Request $request The REST request object containing parameters.
	 *
	 * @return \WP_REST_Response The REST response containing the conditional status and conditions.
	 */
	public function get_conditions( $request ) {
		$required_params = array( 'stepId' );

		// Check if all required parameters are present in the request.
		foreach ( $required_params as $param ) {
			if ( ! isset($request[$param]) ) {
				return rest_ensure_response( $this->prepare_wp_error_response(
					__( "Required parameter '$param' is missing.", 'wpfnl' ), 400
				) );
			}
		}

		// Get the step ID from the request.
		$step_id = sanitize_text_field( $request['stepId'] );

		if( !$step_id ){
			return rest_ensure_response( $this->prepare_wp_error_response(
				__( "Required parameter step id is empty.", 'wpfnl' ), 400
			) );
		}
		
		// Retrieve the conditional status from the post meta.
		$status = get_post_meta( $step_id, '_wpfnl_maybe_enable_condition', true );

		// Prepare the response array.
		$response = [
			'success' => true,
		];

		// Set default status if not available.
		if ( !$status ) {
			$status = 'no';
		}
		$response['status'] = $status;

		// Retrieve step conditions from post meta.
		$conditions = get_post_meta( $step_id, '_wpfnl_step_conditions', true );

		// Ensure conditions are available as an array.
		if ( ! is_array( $conditions ) || ! $conditions ) {
			$conditions = [];
		}
		$response['conditions'] = $conditions;


		$afterCondition = get_post_meta( $step_id, '_wpfnl_next_step_after_condition', true );
		if ( ! is_array( $afterCondition ) || ! $afterCondition ) {
			$afterCondition = [];
		}

		$response['afterCondition'] = $afterCondition;

		// Return a REST response with the status and conditions.
		return rest_ensure_response( $response );
	}
	
	
	
	/**
	 * Save conditional status and conditions for a step.
	 *
	 * This function handles the saving of conditional status and associated conditions for a specific step
	 * based on the provided step ID in the request. It checks for required parameters, sanitizes input,
	 * and updates the post meta with the conditions and after-condition values.
	 *
	 * @since 2.9.0
	 *
	 * @param WP_REST_Request $request The REST request object containing parameters.
	 *
	 * @return \WP_REST_Response The REST response indicating the success of the operation.
	 */
	public function save_condition( $request ) {
		$required_params = array( 'stepId', 'conditions' );

		// Check if all required parameters are present in the request.
		foreach ( $required_params as $param ) {
			if ( ! isset( $request[ $param ] ) ) {
				return rest_ensure_response( $this->prepare_wp_error_response(
					__( "Required parameter '$param' is missing.", 'wpfnl' ), 400
				) );
			}
		}

		$step_id     = sanitize_text_field( $request['stepId'] );
		$conditions  = filter_var_array( $request['conditions'] );

		$afterCondition = filter_var_array( isset( $request['afterCondition'] ) ? $request['afterCondition'] : [] );

		if ( ! $step_id ) {
			return rest_ensure_response( $this->prepare_wp_error_response(
				__( "Required parameter step id is empty.", 'wpfnl' ), 400
			) );
		}

		// Update post meta with the conditions and after-condition values.
		update_post_meta( $step_id, '_wpfnl_step_conditions', $conditions );
		update_post_meta( $step_id, '_wpfnl_next_step_after_condition', $afterCondition );

		// Prepare the response array.
		$response = [
			'success' => true,
		];

		$funnel_id = get_post_meta( $step_id, '_funnel_id', true );
		$reconfigureSettings = get_post_meta( $funnel_id,'_wpfnl_reconfigurable_condition_data', true);
		if( is_array($reconfigureSettings) && !empty($reconfigureSettings) ){
			$key = array_search($step_id, array_column($reconfigureSettings, 'step_id'));
			if( false !== $key ){
				unset($reconfigureSettings[$key]);
				update_post_meta( $funnel_id,'_wpfnl_reconfigurable_condition_data', $reconfigureSettings );
			}
		}
		
		// Return a REST response indicating the success of the operation.
		return rest_ensure_response( $response );
	}
}
