<?php

namespace WPFunnelsPro\Integrations\CRM;
use WPFunnels\Traits\SingletonTrait;
use WPFunnelsPro\Integrations\CRM\Data\CookieData;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

/**
 * CRM integrations for Funnel
 *
 * Class CRM_Integrations
 * @package WPFunnelsPro\Integrations
 * @version 2.0.0
 */
class CrmIntegrations {

    use SingletonTrait;

    private $automation;

    /**
	 * Initialize the class and set its properties.
	 *
	 * @since 2.0.5
	 */
    public function __construct() {
        if (class_exists('WPFunnelsPro\Integrations\CRM')) {
            $this->automation = CookieHandler::getInstance();
            $this->automation->init_actions();
        }
        
        // run scheduler for crm data sending
        add_action( 'wpfunnels/trigger_automation', array( $this, 'run_scheduler' ) );
        add_action( 'wpfunnels/maybe_user_abandoned_funnel', array( $this, 'run_abandoned_funnel_scheduler' ), 10, 3 );
        add_action( 'process_automation_data', array( $this, 'process_automation_data' ), 10, 2 );
        add_action( 'may_be_user_abandoned_funnel', array( $this, 'may_be_user_abandoned_funnel' ) );
    }


    


    /**
     * run scheduler to process data for CRM
     * after funnel is end
     *
     * @param $cookie_data
     */
    public function run_scheduler( $cookie_data ) {

        if ( function_exists('as_has_scheduled_action') ) {
           
            if ( false === as_has_scheduled_action( 'process_automation_data' ) ) {
                $data['data'] = $cookie_data;
                as_schedule_single_action( time(), 'process_automation_data', $data );
            }
        }elseif( function_exists('as_next_scheduled_action') ){
            if ( false === as_next_scheduled_action( 'process_automation_data' ) ) {
                $data['data'] = $cookie_data;
                as_schedule_single_action( time(), 'process_automation_data', $data );
            }
        }
       
        @setcookie( 'wpfunnels_automation_data', null, strtotime( '-1 days' ) );
    }

    /**
     * Process automation data from Cookie and initiate triggers
     * 
     * @param $data
     * @retun null
     */
    public function process_automation_data( $data ) {

        $automations = $this->get_automation_rules($data['funnel_id']);
        if( $automations && !empty($automations) ){
            foreach($automations as $automation){
                if($automation['status'] == 'true'){
                    $cookie_data      = $data;
                    $automation_obj   = new Automation( $automation, $cookie_data );
                    $automation_obj->process_triggers();
                }
            }
        }
    }

    /**
     * Action after user abandoned a funnel
     *
     * @param $step_id
     * @param $funnel_id
     * @param $cookie_data
     */
    public function run_abandoned_funnel_scheduler( $step_id, $funnel_id, $cookie_data ) {
        if ( function_exists('as_has_scheduled_action') ) {
            if ( false === as_has_scheduled_action( 'may_be_user_abandoned_funnel' ) ) {
                $data['data']       = $cookie_data;
                $abandoned_time = apply_filters('wpfunnels/funnel_abandoned_time_limit', strtotime("+30 minutes") );
                as_schedule_single_action( $abandoned_time, 'may_be_user_abandoned_funnel', $data );
            }
        }elseif( function_exists('as_next_scheduled_action') ){
            if ( false === as_next_scheduled_action( 'may_be_user_abandoned_funnel' ) ) {
                $data['data']       = $cookie_data;
                $abandoned_time = apply_filters('wpfunnels/funnel_abandoned_time_limit', strtotime("+30 minutes") );
                as_schedule_single_action( $abandoned_time, 'may_be_user_abandoned_funnel', $data );
            }
        }
    }

    public function may_be_user_abandoned_funnel( $data ) {
        if(isset($data['data'])) {
            $this->process_automation_data($data['data']);
        }
    }

    /**
     * Get offer step id if offer accept or reject
     * 
     * @param $data
     * @param $step_id
     * @return mixed
     */
    private function get_offer_status_from_data( $data, $step_id ) {
        if( isset( $data['offer'] ) && isset($data['offer'][$step_id]) ) {
            return $data['offer'][$step_id];
        }
        return false;
    }

    /**
     * Get Automation Rules data from db
     * 
     * @param $funnel_id
     * @return $automation_rules_data|mixed
     */

    public function get_automation_rules( $funnel_id ) {
        $automation_data = get_post_meta($funnel_id, 'funnel_automation_data', true);
        if( !empty($automation_data)){
            return $automation_data;
        }        
        return false;
    }
}