<?php

namespace WPFunnels\Rest\Controllers;

use DateInterval;
use DateTime;
use WP_Error;
use WP_REST_Request;
use WP_REST_Server;
use WPFunnels\Wpfnl_functions;
use WPFunnelsProGBF\Conditions;
use WPFunnelsProGBF\Conditions\Date;
use WPFunnelsPro\ReplaceOrder\OrderReplacement;

class GlobalFunnelController extends Wpfnl_REST_Controller
{
    
    /**
     * Endpoint namespace.
     *
     * @var string
     */
    protected $namespace = 'wpfunnels/v1';

    /**
     * Route base.
     *
     * @var string
     */
    protected $rest_base = 'global-funnel';


    /**
     * check if user has valid permission
     *
     * @param $request
     * @return bool|WP_Error
     * @since 1.0.0
     */
    public function update_items_permissions_check($request)
    {   
        return true;
        if (!Wpfnl_functions::wpfnl_rest_check_manager_permissions( 'step', 'edit' )) {
            return new WP_Error('wpfunnels_rest_cannot_edit', __('Sorry, you cannot edit this resource.', 'wpfnl'), ['status' => rest_authorization_required_code()]);
        }
        return true;
    }

    /**
     * Makes sure the current user has access to READ the settings APIs.
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|boolean
     * @since  1.0.0
     */
    public function get_items_permissions_check($request)
    {
        if (!Wpfnl_functions::wpfnl_rest_check_manager_permissions('settings')) {
            return new WP_Error('wpfunnels_rest_cannot_view', __('Sorry, you cannot list resources.', 'wpfnl'), ['status' => rest_authorization_required_code()]);
        }
        return true;
    }

    /**
     * register rest routes
     *
     * @since 1.0.0
     */
    public function register_routes()
    {
        register_rest_route($this->namespace, '/' . $this->rest_base . '/change-trigger-type'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => WP_REST_Server::EDITABLE,
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => array( $this, 'change_trigger_type'),
                'permission_callback' => array( $this, 'update_items_permissions_check' ) ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => array( $this, 'get_global_funnel_data'),
                'permission_callback' => array( $this, 'update_items_permissions_check' ) ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/saveSettings'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'POST',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'save_settings'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);


        register_rest_route($this->namespace, '/' . $this->rest_base . '/get_order_bump_rules'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_order_bump_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/get_steps'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_funnel_steps'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/get_highest_sale_products', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_highest_sale_products'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);


        /**
         * start according to updated plan
         */

        register_rest_route($this->namespace, '/' . $this->rest_base . '/updateStatus'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'POST',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'update_status'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);


        register_rest_route($this->namespace, '/' . $this->rest_base . '/getStatus'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_status'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);


        register_rest_route($this->namespace, '/' . $this->rest_base . '/saveStartCondition'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'POST',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'save_start_condition'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/getStartConditions'. '/(?P<funnel_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'funnel_id' => array(
                        'description'   => __('Funnel ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_start_conditions'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/saveObRules'. '/(?P<step_id>\d+)', [
            array(
                'methods' => 'POST',
                'args' => array(
                    'step_id' => array(
                        'description'   => __('Step ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'save_ob_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/getObRules'. '/(?P<step_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'step_id' => array(
                        'description'   => __('Step ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_ob_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/saveUpsellRules'. '/(?P<step_id>\d+)', [
            array(
                'methods' => 'POST',
                'args' => array(
                    'step_id' => array(
                        'description'   => __('Step ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'save_upsell_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/saveDownsellRules'. '/(?P<step_id>\d+)', [
            array(
                'methods' => 'POST',
                'args' => array(
                    'step_id' => array(
                        'description'   => __('Step ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'save_downsell_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/getUpsellRules'. '/(?P<step_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'step_id' => array(
                        'description'   => __('Step ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_upsell_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        register_rest_route($this->namespace, '/' . $this->rest_base . '/getDownsellRules'. '/(?P<step_id>\d+)', [
            array(
                'methods' => 'GET',
                'args' => array(
                    'step_id' => array(
                        'description'   => __('Step ID.', 'wpfnl'),
                        'type'          => 'string',
                    )
                ),
                'callback' => [
                    $this,
                    'get_downsell_rules'
                ],
                'permission_callback' => [
                    $this,
                    'update_items_permissions_check'
                ] ,
            ),
        ]);

        // end according to updated plan

        
    }
    //start according to updated plan

    /**
     * Update global funnel status by funnel_id
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     * 
     */
    public function update_status( \WP_REST_Request $request ){
        $funnel_id       = isset($request['funnel_id']) ? $request['funnel_id'] : '';
        $data = 'no';
        if( isset($request['data']) ){
            $data = $request['data'] == 'true' ? 'yes' : 'no';
        }
        $response = array(
            'success' => false,
            'message' => 'Fail'
        );
        if( $funnel_id ){
            update_post_meta( $funnel_id, 'is_global_funnel', $data );
            $response = array(
                'success' => true,
                'data'    => $request['data']
            );
        }
        return rest_ensure_response($response);

    }
    
    
    /**
     * get global funnel status by funnel_id
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     * 
     */
    public function get_status( \WP_REST_Request $request ){
        $funnel_id       = isset($request['funnel_id']) ? $request['funnel_id'] : '';
        $response = array(
            'success' => false,
            'data' => 'Fail'
        );
        if( $funnel_id ){
            $data = get_post_meta( $funnel_id, 'is_global_funnel', true );
            $response = array(
                'success' => true,
                'data'    => $data == 'yes' ? true : false
            );
        }
        return rest_ensure_response($response);

    }

    /**
     * Save global funnel start condition to postmeta by funnel_id
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function save_start_condition( \WP_REST_Request $request ){

        $response = array(
            'success' => false,
            'message' => 'Fail'
        ); 
        if( !isset($request['funnel_id']) || !isset($request['data'])){
            return rest_ensure_response($response);
        }
        $funnel_id       = isset($request['funnel_id']) ? $request['funnel_id'] : '';
        $start_condition = isset($request['data']['conditions']) && !empty($request['data']['conditions']) ? $request['data']['conditions'] :  array();
        $step_id         = isset($request['data']['step_id']) ? $request['data']['step_id'] : '';
        $coupon          = isset($request['data']['coupon']) && $request['data']['coupon'] == 'true' ? 'yes' : 'no';
        $isQuantity      = isset($request['data']['isQuantity']) && $request['data']['isQuantity'] == 'true' ? 'yes' : 'no';
       
        if( isset($request['data']['enableAutoCoupon'], $request['data']['selectedCoupon'] ) ){
			$auto_coupon = [
				'enableAutoCoupon' => $request['data']['enableAutoCoupon'],
				'selectedCoupon' => $request['data']['selectedCoupon'],
			];
           
			update_post_meta( $step_id, '_wpfnl_checkout_auto_coupon', $auto_coupon );
		}

        if( isset($request['data']['discountOptions']) ){
			$discount = [
				'discountOptions' 	 => isset($request['data']['discountOptions']) ? $request['data']['discountOptions'] : 'original',
				'discountapplyto' 	 => isset($request['data']['discountapplyto']) && $request['data']['discountOptions'] != 'original' ? $request['data']['discountapplyto'] : '',
				'mutedDiscountValue' => isset($request['data']['mutedDiscountValue']) && $request['data']['discountOptions'] != 'original' ? $request['data']['mutedDiscountValue'] : 0,
				'discountedPrice'    => isset($request['data']['discountedPrice']) && $request['data']['discountOptions'] != 'original' ? $request['data']['discountedPrice'] : 0,
			];
            if( $step_id ){
			    update_post_meta($step_id, '_wpfnl_checkout_discount_main_product', $discount);
            }
		}

        
        if( $step_id ){
            update_post_meta($step_id, '_wpfnl_checkout_coupon', $coupon);
		    update_post_meta($step_id, '_wpfnl_quantity_support', $isQuantity);
        }
        
        
		

        if( $funnel_id && !empty($start_condition) ){
            update_post_meta( $funnel_id, 'global_funnel_start_condition', $start_condition );
            $response = array(
                'success' => true,
                'message' => 'Save successful'
            ); 
        }
        return rest_ensure_response($response);
    }
    
    
    /**
     * Get global funnel start condition from postmeta by funnel_id
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     * 
     */
    public function get_start_conditions( \WP_REST_Request $request ){

        $funnel_id       = isset($request['funnel_id']) ? $request['funnel_id'] : '';
        $step_id         = isset($request['step_id']) ? $request['step_id'] : '';
        $response = array(
            'success' => false,
            'data'    => 'Data not found'
        );
        if( !$funnel_id || !$step_id ){
            return rest_ensure_response($response);
        }
        $start_condition = get_post_meta( $funnel_id, 'global_funnel_start_condition', true );
        
        $discount 			=  get_post_meta($step_id, '_wpfnl_checkout_discount_main_product', true);
        $auto_coupon        =  get_post_meta($step_id, '_wpfnl_checkout_auto_coupon', true);
        $dateTime = new DateTime();
        if( empty($start_condition) ){
            $start_condition = array(
                array(
                    'param' => '',
                    'operator' => '==',
                    'name' => '', 
                    'value' => '', 
                    'date' => $dateTime->format('M d, Y'),
                    'startDate' => $dateTime->format('M d, Y'),
                    'endDate' => $dateTime->add(new DateInterval('P1D'))->format('M d, Y')
                )
            ); 
            
        }else{
            if( is_array($start_condition) ){
                foreach( $start_condition as $key=>$condition ){
                    $start_condition[$key]['date'] = isset($start_condition[$key]['date']) && $start_condition[$key]['date'] ? $start_condition[$key]['date'] : $dateTime->format('M d, Y');
                    $start_condition[$key]['startDate'] = isset($start_condition[$key]['startDate']) && $start_condition[$key]['startDate'] ? $start_condition[$key]['startDate'] : $dateTime->format('M d, Y');
                    $start_condition[$key]['endDate'] = isset($start_condition[$key]['endDate']) && $start_condition[$key]['endDate'] ? $start_condition[$key]['endDate'] : $dateTime->add(new DateInterval('P1D'))->format('M d, Y');
                }
            }
            
        }

        $coupon          = get_post_meta( $step_id, '_wpfnl_checkout_coupon', true );
        $isQuantity      = get_post_meta( $step_id, '_wpfnl_quantity_support', true );
        $data = array(
            'start_condition' => $start_condition,
            'coupon' => $coupon,
            'autoCoupon' => $auto_coupon,
            'isQuantity' => $isQuantity,
            'discount' => $discount,
        );
        if(  !empty($start_condition) ){
            $response = array(
                'success'   => true,
                'data'      => $data
            ); 
        }
        return rest_ensure_response($response);
    }


    /**
     * Save global funnel order bump rules to postmeta by step id
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     * 
     */
    public function save_ob_rules( \WP_REST_Request $request ){
        $step_id       = isset($request['step_id']) ? $request['step_id'] : '';
        $ob_rules      = isset($request['data']) ? $request['data'] : '';
        $response = array(
            'success' => true,
            'message' => 'false'
        ); 

        if( !$step_id || !$ob_rules ){
            return rest_ensure_response($response);
        }

        update_post_meta( $step_id, 'global_funnel_ob_rules', $ob_rules );
        $response = array(
            'success' => true,
            'message' => 'Save successful'
        ); 
        return rest_ensure_response($response);
    }


    /**
     * Get global funnel order bump rules from postmeta by funnel_id
     * 
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     * 
     */
    public function get_ob_rules( \WP_REST_Request $request ){

        $step_id  = isset($request['step_id']) ? $request['step_id'] : '';
        $ob_rules   = get_post_meta( $step_id, 'global_funnel_ob_rules', true );
        $response   = array(
            'success'     => false,
            'data'        => 'Data not found',
        );
        if(  !empty($ob_rules) ){
            $response = array(
                'success'   => true,
                'data'      => $ob_rules
            ); 
        }
        return rest_ensure_response($response);
    }


    /**
     * Save global funnel upsell rules to postmeta by step_id
     *
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */

    public function save_upsell_rules(\WP_REST_Request $request){


        $step_id = isset($request['step_id']) ? $request['step_id'] : '' ;
        $upsell_rules = isset($request['data'])? $request['data'] : '';
        $type = isset($request['type'])? $request['type'] : 'upsell';
        $product = isset($upsell_rules['show']) && $upsell_rules['show'] ?  wc_get_product( $upsell_rules['show'] ) : '';

        $upsell_rules['price'] = $product ? $product->get_price_html() : '';
        $upsell_rules['sale_price'] = $product ? wc_price($product->get_sale_price()) : '';
        $upsell_rules['regular_price'] = $product ? wc_price($product->get_regular_price()) : '';
        
        $response = array(
            'success' => false,
            'message' => 'Fail',
        ) ;
        
        if( $step_id && $upsell_rules ){
            update_post_meta($step_id,'global_funnel_upsell_rules',$upsell_rules);
            $discount = isset($request['discount']) ? $request['discount'] : [];
            if( !empty($discount) ){
                update_post_meta( $step_id, '_wpfnl_'.$type.'_discount', $discount );
            }
            
            $step_info = array(
                'step_id'   => $step_id,
                'step_type' => $type,
            );
    
            if( isset($request['replaceSettings']) && 'true' === $request['replaceSettings'] ){
                update_post_meta( $step_info['step_id'], '_wpfnl_'.$step_info['step_type'].'_replacement_settings', 'true' );
                $offerReplacement = $request['offerReplacement'];
                $replacement_type = isset($offerReplacement['replacement_type']) && $offerReplacement['replacement_type'] ? $offerReplacement['replacement_type'] : '';
                $value = isset($offerReplacement['value']) && $offerReplacement['value'] ? $offerReplacement['value'] : '';
                OrderReplacement::save_replace_order_condition( $step_info, $replacement_type, $value );
            }else{
                update_post_meta( $step_info['step_id'], '_wpfnl_'.$step_info['step_type'].'_replacement_settings', 'false' );
                delete_post_meta( $step_info['step_id'], '_wpfnl_'.$step_info['step_type'].'_replacement' );
            }

            $response = array(
                'success' => true,
                'message' => 'Save Successful',
            ) ;
        }
        
        return rest_ensure_response($response);
    }


    /**
     * Get upsell rules from postmeta by step_id
     * 
     * @param WP_REST_Request $request
     * @return array|string[]
     */
    public function get_upsell_rules(\WP_REST_Request $request)
    {
        $response   = array(
            'success' => false,
            'data'    => 'Data not found',
            'prevStep'        => '',
            'isChildOrder'    => false,
            'discount'    => array(
                                'discountType'      => 'original',
                                'discountApplyTo'   => 'regular',
                                'discountValue'     => '0',
                                'discountPrice'     => '',
                                'discountPriceHtml' => '',
                            ),
            'columns'    => [],
        );


        $step_id  = isset($request['step_id']) ? $request['step_id'] : '';

        if( $step_id ){
            $upsell_rules   = get_post_meta( $step_id, 'global_funnel_upsell_rules', true );
            
            $discount   = get_post_meta( $step_id, '_wpfnl_upsell_discount', true );
            /** if offer product has discount */
            if( $discount ) {
                $discount_type     = $discount[ 'discountType' ];
                $discount_apply_to = $discount[ 'discountApplyTo' ];
                $discount_value    = $discount[ 'discountValue' ];
                $discount[ 'discountPrice' ] = '';
                $discount[ 'discountPriceHtml' ] = '';
            }

            if(  !empty($upsell_rules) ){
                $offer_settings             = Wpfnl_functions::get_offer_settings();
                $funnel_id                  = Wpfnl_functions::get_funnel_id_from_step( $step_id );
                $prev_step                  = Wpfnl_functions::get_prev_step( $funnel_id, $step_id );
                $response = array(
                    'success'       => true,
                    'data'          => $upsell_rules,
                    'discount'      => $discount ? $discount : array(
                                                                'discountType'      => 'original',
                                                                'discountApplyTo'   => 'regular',
                                                                'discountValue'     => '0',
                                                                'discountPrice'     => '',
                                                                'discountPriceHtml' => '',
                                                            )
                );

                $replaceSettings = get_post_meta( $step_id, '_wpfnl_upsell_replacement_settings', true );
                if( $replaceSettings == 'true' ) {
                    $response[ 'replaceSettings' ] = $replaceSettings;
                    $isOfferReplace                = get_post_meta( $step_id, '_wpfnl_upsell_replacement', true );
                    $response[ 'isOfferReplace' ]  = array(
                        'replacement_type' => $isOfferReplace[ 'replacement_type' ],
                        'value'            => $isOfferReplace[ 'value' ],
                    );
                }
                else {
                    $response[ 'replaceSettings' ] = $replaceSettings;
                    $response[ 'isOfferReplace' ]  = array(
                        'replacement_type' => '',
                        'value'            => '',
                    );
                }
                $response[ 'prevStep' ]     = isset( $prev_step[ 'step_type' ] ) && $prev_step[ 'step_type' ] ? $prev_step[ 'step_type' ] : '';
                $response[ 'isChildOrder' ] = isset($offer_settings[ 'offer_orders' ]) && ($offer_settings[ 'offer_orders' ] == 'child-order') ? true : false;
                $response[ 'columns' ]      = Wpfnl_functions::get_checkout_columns( $step_id );
            }
        }
        
        return rest_ensure_response($response);

    }

    /**
     * Save global funnel downsell rules to postmeta by step_id
     *
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function save_downsell_rules(\WP_REST_Request $request){
        
        $response = array(
            'success' => false,
            'message' => 'Fail',
        ) ;
        
        
        $step_id = isset($request['step_id']) ? $request['step_id'] : '' ;
        $downsell_rules = isset($request['data'])? $request['data'] : '';
        $type = isset($request['type'])? $request['type'] : 'downsell';
        if( !$downsell_rules ){
            return rest_ensure_response($response);
        }

        $product = wc_get_product( $downsell_rules['show'] );
        if( $product ){
            $downsell_rules['price'] = $product->get_price_html();
            $downsell_rules['sale_price'] = wc_price($product->get_sale_price());
            $downsell_rules['regular_price'] = wc_price($product->get_regular_price());
        }else{
            $downsell_rules['price'] = '';
            $downsell_rules['sale_price'] = '';
            $downsell_rules['regular_price'] = '';
        }
        
        update_post_meta($step_id,'global_funnel_downsell_rules',$downsell_rules);

        $discount = isset($request['discount']) ? $request['discount'] : [];
        
        if( !empty($discount) ){
            update_post_meta( $step_id, '_wpfnl_'.$type.'_discount', $discount );
        }
        
        $step_info = array(
            'step_id'   => $step_id,
            'step_type' => $type,
        );

        if( isset($request['replaceSettings']) && 'true' === $request['replaceSettings'] ){
            update_post_meta( $step_info['step_id'], '_wpfnl_'.$step_info['step_type'].'_replacement_settings', 'true' );
            $offerReplacement = $request['offerReplacement'];
            $replacement_type = isset($offerReplacement['replacement_type']) && $offerReplacement['replacement_type'] ? $offerReplacement['replacement_type'] : '';
            $value = isset($offerReplacement['value']) && $offerReplacement['value'] ? $offerReplacement['value'] : '';
            OrderReplacement::save_replace_order_condition( $step_info, $replacement_type, $value );
        }else{
            update_post_meta( $step_info['step_id'], '_wpfnl_'.$step_info['step_type'].'_replacement_settings', 'false' );
            delete_post_meta( $step_info['step_id'], '_wpfnl_'.$step_info['step_type'].'_replacement' );
        }
        $response = array(
            'success' => true,
            'message' => 'Save Successful',
        ) ;
       
        return rest_ensure_response($response);
    }


    /**
     * Get downsell rules from postmeta by step_id
     * 
     * @param WP_REST_Request $request
     * @return array|string[]
     */
    public function get_downsell_rules( \WP_REST_Request $request )
    {
        $step_id  = isset($request['step_id']) ? $request['step_id'] : '';
        $downsell_rules   = get_post_meta( $step_id, 'global_funnel_downsell_rules', true );
        $response   = array(
            'success' => false,
            'data'    => 'Data not found',
            'prevStep'        => '',
            'isChildOrder'    => false,
            'discount'    => array(
                                'discountType'      => 'original',
                                'discountApplyTo'   => 'regular',
                                'discountValue'     => '0',
                                'discountPrice'     => '',
                                'discountPriceHtml' => '',
                            ),
            'columns'    => [],
        );
        if( $step_id && !empty($downsell_rules) ){
            $offer_settings             = Wpfnl_functions::get_offer_settings();
            $funnel_id                  = Wpfnl_functions::get_funnel_id_from_step( $step_id );
            $prev_step                  = Wpfnl_functions::get_prev_step( $funnel_id, $step_id );
            $discount   = get_post_meta( $step_id, '_wpfnl_downsell_discount', true );
            /** if offer product has discount */
            if( $discount ) {
                $discount_type     = $discount[ 'discountType' ];
                $discount_apply_to = $discount[ 'discountApplyTo' ];
                $discount_value    = $discount[ 'discountValue' ];
                $discount[ 'discountPrice' ] = '';
                $discount[ 'discountPriceHtml' ] = '';
            }

            $response = array(
                'success'   => true,
                'data'      => $downsell_rules,
                'discount'      => $discount ? $discount : array(
                    'discountType'      => 'original',
                    'discountApplyTo'   => 'regular',
                    'discountValue'     => '0',
                    'discountPrice'     => '',
                    'discountPriceHtml' => '',
                )
            );

            $replaceSettings = get_post_meta( $step_id, '_wpfnl_downsell_replacement_settings', true );
            if( 'true' === $replaceSettings ) {
                $response[ 'replaceSettings' ] = $replaceSettings;
                $isOfferReplace                = get_post_meta( $step_id, '_wpfnl_downsell_replacement', true );
                $response[ 'isOfferReplace' ]  = array(
                    'replacement_type' => $isOfferReplace[ 'replacement_type' ],
                    'value'            => $isOfferReplace[ 'value' ],
                );
            }
            else {
                $response[ 'replaceSettings' ] = $replaceSettings;
                $response[ 'isOfferReplace' ]  = array(
                    'replacement_type' => '',
                    'value'            => '',
                );
            }
            $response[ 'prevStep' ]     = isset( $prev_step[ 'step_type' ] ) && $prev_step[ 'step_type' ] ? $prev_step[ 'step_type' ] : '';
            $response[ 'isChildOrder' ] = isset($offer_settings[ 'offer_orders' ]) && ($offer_settings[ 'offer_orders' ] == 'child-order') ? true : false;
            $response[ 'columns' ]      = Wpfnl_functions::get_checkout_columns( $step_id );

        }
        return rest_ensure_response($response);

    }

    

    // end according to updated plan


    /**
     * update trigger type
     *
     * @param WP_REST_Request $request
     * @return WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function change_trigger_type(\WP_REST_Request $request) {
        $funnel_id 	= $request['funnel_id'];
        if( !$funnel_id ) {
            $response['success'] = false;
            return rest_ensure_response($response);
        }
        $settings = $request->get_params();
        if( isset($settings['type']) ) {
            wpfnl()->meta->update_meta( $funnel_id, 'global_funnel_trigger_type', $settings['type'] );
        }
        $response['success'] = true;
        return rest_ensure_response($response);
    }


    /**
     * get global funnel data
     *
     * @param WP_REST_Request $request
     * @return false|WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_global_funnel_data( \WP_REST_Request $request ) {
        $funnel_id 	= $request['funnel_id'];
        if( !$funnel_id ) {
            return false;
        }
        $trigger_type           = $this->get_global_funnel_trigger( $funnel_id );
        $start_condition        = $this->get_start_condition( $funnel_id, $trigger_type );

        $order_bump_rule_type   = $this->get_rule_type( $funnel_id, 'ob' );
        $order_bump_rules       = $this->get_rules( $funnel_id, 'ob' );

        $upsell_rule_type       = $this->get_rule_type( $funnel_id, 'upsell' );
        $upsell_rules           = $this->get_rules( $funnel_id, 'upsell' );

        $downsell_rule_type     = $this->get_rule_type( $funnel_id, 'downsell' );
        $downsell_rules         = $this->get_rules( $funnel_id, 'downsell' );
        $data = array(
            'trigger_type'      => $trigger_type,
            'start_condition'   => $start_condition,
            'ob_rule_type'      => $order_bump_rule_type,
            'ob_rules'          => $order_bump_rules,
            'upsell_rule_type'  => $upsell_rule_type,
            'upsell_rules'      => $upsell_rules,
            'downsell_rule_type'=> $downsell_rule_type,
            'downsell_rules'    => $downsell_rules,
        );
        return rest_ensure_response( $data );
    }




    /**
     * get global funnel trigger type
     *
     * @param $funnel_id
     * @return string
     */
    private function get_global_funnel_trigger( $funnel_id ) {
        return wpfnl()->meta->get_funnel_meta( $funnel_id, 'global_funnel_trigger_type', 'category' );
    }


    /**
     * get start condition
     *
     * @param $funnel_id
     * @param string $type
     * @return array
     */
    private function get_start_condition( $funnel_id, $type = 'category' ) {
        $default_value = array(
            array(
                array(
                    'param'     => 'category',
                    'operator'  => '==',
                    'value'     => '0',
                    'name'      => '',
                )
            )
        );
        $start_condition            = wpfnl()->meta->get_funnel_meta( $funnel_id, 'global_funnel_start_condition', $default_value );
        if($start_condition) {
            foreach ($start_condition as $key => $condition_group) {
                
                if ( empty( $condition_group ) ) {
                    continue;
                }
                foreach ( $condition_group as $k => $rule ) {
                    
                    if('category' === $type) {
                        $name = $this->get_name_by_term_id( $rule['value'], 'product_cat' );
                        $start_condition[$key][$k]['name'] = $name;

                    }
                }
            }
        }
        return $start_condition;
    }


    /**
     * get term name by id
     *
     * @param $term_id
     * @param $taxonomy
     * @return string
     */
    private function get_name_by_term_id( $term_id, $taxonomy ){
        $get_term = get_term_by( 'id', $term_id, $taxonomy );
        if( !$get_term ){
            return 'not_found';
        }
        return  $get_term->name;
    }


    /**
     * get rule type
     *
     * @param $funnel_id
     * @param $type
     * @return string
     */
    private function get_rule_type( $funnel_id, $type ) {
        return wpfnl()->meta->get_funnel_meta( $funnel_id, "global_funnel_{$type}_rules_type", 'map' );
    }

    /**
     * Get funnel steps by funnel id
     * @param WP_REST_Request $request
     * @return false|WP_Error|\WP_HTTP_Response|\WP_REST_Response
     */
    public function get_funnel_steps( \WP_REST_Request $request ){
        $funnel_id 	= $request['funnel_id'];
        if( !$funnel_id ) {
            return false;
        }

        $steps = Wpfnl_functions::get_steps( $funnel_id );
        
        $offer_steps = array();
        $i = 0;
        // remove landing and thankyou steps
        foreach($steps as $key=>$step){
            if( $step['step_type'] == 'thankyou' || $step['step_type'] == 'landing' ){
                unset($steps[$key]);
            }else{
                $offer_steps[$i]['id'] = $step['id'];
                $offer_steps[$i]['step_type'] = $step['step_type'];
                $offer_steps[$i]['name'] = $step['step_type'] != 'checkout' ? $step['name'] : 'Order Bump';
                $i++;
            }
        }

        return rest_ensure_response( $offer_steps );

    }


    /**
     * get order bump rules
     *
     * @param $funnel_id
     * @param $type
     * @return mixed
     */
    public function get_rules( $funnel_id, $type )
    {
        $default_value = array(
            array(
                array(
                    'param'         => 'product',
                    'operator'      => '==',
                    'type'          => '',
                    'value'         => '',
                    'valueName'     => '',
                    'show'          => '',
                    'showName'      => '',
                ),
            ),
        );
        $rule_groups = wpfnl()->meta->get_funnel_meta( $funnel_id, "global_funnel_{$type}_rules", $default_value );
        
        if($rule_groups) {
            foreach ($rule_groups as $key => $condition_group) {
                if ( empty( $condition_group ) || !is_array($condition_group)) {
                    continue;
                }
                foreach ( $condition_group as $k => $rule ) {
                    /** get value product name */
                    if('product' === $rule['param']) {
                        $name                               = $rule['value'] ? $this->get_name_by_product_id( $rule['value'] ) : '';
                        $rule_groups[$key][$k]['valueName'] = $name;
                    }
                    /** get show product name */
                    $name                               = $rule['show'] ? $this->get_name_by_product_id( $rule['show'] ) : '';
                    $rule_groups[$key][$k]['showName']  = $name;
                    $type                               = isset($rule['type']) ? $rule['type'] : '';
                    $rule_groups[$key][$k]['type']  = $type;
                }
            }
        }
        return $rule_groups;
    }


    /**
     * get product name by product id
     *
     * @param $product_id
     * @return string
     */
    private function get_name_by_product_id( $product_id ) {
        return get_the_title( $product_id );
    }


//    private function get_start_condition( $funnel_id, $type = 'category' ) {
//        $group_conditions   = wpfnl()->meta->get_funnel_meta( $funnel_id, 'global_funnel_start_condition' );
//        $trigger_meta       = array();
//        if( $group_conditions ) {
//            foreach( $group_conditions as $group ){
//                if ( empty( $group ) ) {
//                    continue;
//                }
//                foreach ( $group as $condition ){
//                    if('category' === $type) {
//                        $term_id        = $condition['value'];
//                        $trigger_meta[] = $this->get_global_funnel_trigger_meta($term_id, 'product_cat');
//                    }
//                    $get_term = $this->get_name_by_term_id($category_id);
//                    $category_data = [
//                        'param' => $value['param'],
//                        'operator' =>$value['operator'],
//                        'value' =>array(
//                            'id' => $category_id,
//                            'name' => $get_term['name'],
//                            'slug' => $get_term['slug']
//                        )
//                    ];
//                }
//            }
//        }
//    }


    private function get_global_funnel_trigger_meta( $term_id, $taxonomy ) {
        $get_term = get_term_by('id', $term_id, $taxonomy);
        if( !$get_term ){
            return 'not found';
        }
        $term = array(
            'name' => $get_term->name,
            'slug' => $get_term->slug,
        );
    }

    /**
     * Save global settings
     * 
     * @param WP_REST_Request $request Full data about the request.
     * 
     * @since 1.0.0
     */
    public function save_settings( \WP_REST_Request $request ){

        $funnel_id 	= $request['funnel_id'];

        $data = array(
            'start_condition'       => $request['start_condition'],
            'ob_rules'              => $request['ob_rules'],
            'upsell_rules'          => $request['upsell_rules'],
            'downsell_rules'        => $request['downsell_rules'],
            'ob_rules_type'         => $request['ob_rules_type'],
            'upsell_rules_type'     => $request['upsell_rules_type'],
            'downsell_rules_type'   => $request['downsell_rules_type'],
        );
        
        foreach ( $data as $key => $value ){
            update_post_meta( $funnel_id,'global_funnel_'.$key,$value );
        }

        $response['success'] = true;
        return $response;
    }



    /**
     * Get highest sale products
     * @param WP_REST_Request $request
     * @return array|string[]
     */
    public function get_highest_sale_products(){
        global $wpdb;
    
        $limit_clause = intval(5) <= 0 ? '' : 'LIMIT '. intval(5);
        $product_ids = array();
        $product_ids = $wpdb->get_results("
                        SELECT p.ID as id, COUNT(oim2.meta_value) as count
                        FROM {$wpdb->prefix}posts p
                        INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim
                            ON p.ID = oim.meta_value
                        INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta oim2
                            ON oim.order_item_id = oim2.order_item_id
                        INNER JOIN {$wpdb->prefix}woocommerce_order_items oi
                            ON oim.order_item_id = oi.order_item_id
                        INNER JOIN {$wpdb->prefix}posts as o
                            ON o.ID = oi.order_id
                        WHERE p.post_type = 'product'
                        AND p.post_status = 'publish'
                        AND oim.meta_key = '_product_id'
                        GROUP BY p.ID
                        ORDER BY COUNT(oim2.meta_value) + 0 DESC
                        $limit_clause
                    ");
        $formatted_data = array();
        $i = 1;
        foreach( $product_ids as $product_id ){
            $ext = 'th';
            if( $i == 1 ){
                $ext = 'st';
            }
            if( $i == 2 ){
                $ext = 'nd';
            }
            if( $i == 3 ){
                $ext = 'rd';
            }
            $formatted_data[$product_id->id] = $i. $ext. ' highest';
            $i++;
        }
        return rest_ensure_response( $formatted_data );
    }


}