<?php
namespace WPFunnelsProGBF\Conditions\Offer;

/**
 * @author [Getwpfunnels Team]
 * @email [support@getwpfunnels.com]
 * @create date 2022-06-21 17:29:35
 * @modify date 2022-06-21 17:29:35
 * @desc [Get offer product when GBF condition on upsell and downsell is "Random product in category"]
 */

use WPFunnelsPro\Wpfnl_Pro_functions;
use Wpfnl_Pro_GB_Functions;

class RandomProduct extends Wpfnl_Pro_GBF_Conditions {

    /**
     * Get dynamic offer product for upsell or downsell
     * Offer product will be random product of a catefory
     * @param mixed $offer_mappings
     * @param mixed $order_id
     * @param mixed $step_id
     * @param mixed $gbf_product
     * 
     * @return array
     * 
     * @since 1.0.15
     */
    public function get_offer_product( $offer_mappings, $order_id, $step_id, $gbf_product ) {
        $show          = isset($offer_mappings['show']) ? $offer_mappings['show'] : null;
        $category      = isset($offer_mappings['category']) ? $offer_mappings['category'] : null;
        $quantity      = isset($offer_mappings['quantity']) ? $offer_mappings['quantity'] : null;

        if( !get_option( 'wpfunnels_dynamic_offer_data' ) ){      
            Wpfnl_Pro_GB_Functions::get_random_product_in_category_for_offer( 1, $category );
        }

        // Fetching offer product from option table `wpfunnels_dynamic_offer_data`
        $dynamic_product = $this->fetch_offer_product();

        // If the dynamic product is not valid returns empty array
        if( !is_array($dynamic_product) || !isset($dynamic_product[0]['ID'])) {
            return [];
        }
        // Updating the value of the offer product's id on $show by checking if the offer product already ordered
        $show = $this->get_show_by_handling_dynamic_offer_data( $dynamic_product, $order_id, $category, $show );

        return Wpfnl_Pro_functions::get_offer_data( $step_id, $show, $quantity, $order_id );
    }

    /**
     * Fetches the offer product data.
     * Retrieves and decodes the offer product data stored in the 'wpfunnels_dynamic_offer_data' option.
     * 
     * @return array|false The offer product data as an associative array, or false if the option is not set or cannot be decoded.
     * 
     * @since 2.7.13
     */
    public function fetch_offer_product(){
        return json_decode( wp_unslash( get_option( 'wpfunnels_dynamic_offer_data' ) ), true );
    }

    /**
     * Check if a dynamic product is already ordered in a given order.
     *
     * Retrieves the order object based on the provided order ID and checks if the dynamic product is already present in the order.
     *
     * @param int $order_id The ID of the order to check.
     * @param array $dynamic_product The dynamic product data to check against.
     * @return bool True if the dynamic product is already ordered, false otherwise.
     * 
     * @since 2.7.13
     */
    public function is_already_ordered($order_id , $dynamic_product){
        $order = wc_get_order($order_id);

        // Checks dynamic product already in order or not
        if( !empty($order) ){
            foreach ( $order->get_items() as $item_id => $item ) {
                $product_id = $item->get_variation_id() ? intval($item->get_variation_id()) : intval($item->get_product_id());
                $dynamic_product_id = intval($dynamic_product[0]['ID']);
                if( $dynamic_product_id === $product_id ){
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Gets the show value by handling dynamic offer data.
     *
     * @param array  $dynamic_product The dynamic product data.
     * @param mixed  $order_id The order ID.
     * @param mixed  $category The category value.
     * @param mixed  $show The initial show value.
     * @return mixed The updated show value.
     *
     * @since 2.7.13
     */
    private function get_show_by_handling_dynamic_offer_data( $dynamic_product, $order_id, $category, $show ){
        if( is_array($dynamic_product) && isset($dynamic_product[0]['ID'])) {
            if( !$this->is_already_ordered( $order_id, $dynamic_product ) ){
                // Offered product is not already ordered
                $show = $dynamic_product[0]['ID'];  
            } else {
                // If dynamic product already in order then regenerate dynamic product agian
                $this->delete_dynamic_offer_data();
                Wpfnl_Pro_GB_Functions::get_random_product_in_category_for_offer( 1, $category );
                $dynamic_product = $this->fetch_offer_product();
                if( is_array($dynamic_product) && isset($dynamic_product[0]['ID'])){
                    $show = $dynamic_product[0]['ID'];  
                }
            }
        } else {
            $this->delete_dynamic_offer_data();
        }
        return $show;
    }

    /**
     * Deletes dynamic offer data option.
     *
     * This function deletes the option 'wpfunnels_dynamic_offer_data'.
     *
     * @since 2.7.13
     *
     * @return void
     */
    private function delete_dynamic_offer_data(){
        delete_option( 'wpfunnels_dynamic_offer_data' );
    }

}